/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:    All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.    The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, bitwise: true, node: true, expr: true*/
/*global describe: true, _: true, expect: true, window: true, WebSocket: true, it: true, afterEach */
"use strict";

var FileUtils = require("../lib/file-utils"),
    fs = require("fs-extra"),
    sinon = require("sinon"),
    Q = require("q");

describe("FileUtils", function() {
    var sandbox = sinon.sandbox.create();

    afterEach(function () {
        sandbox.restore();
    });

    var FakeFS = {
        copy: function(src, dest, callback) { callback(); },
        mkdirs: function(path, callback) { callback(); },
        rename: function(src, dest, callback) { callback(); },
        remove: function(path, callback) { callback(); },
        // Failing variants.
        failingRename: function(src, dest, callback) { callback("error"); },
        failingCopy: function(src, dest, callback) { callback("error"); },
    };

    it("when moving a file, should make directories", function() {
        sandbox.stub(fs, "mkdirs", FakeFS.mkdirs);
        sandbox.stub(fs, "rename", FakeFS.rename);

        var promise = FileUtils.moveTemporaryFile("/tmp/123", "/Users/Batman/Desktop/batman.png");
        return expect(promise).to.eventually.be.fulfilled.then(function() {
            expect(fs.mkdirs).to.have.been.calledWith("/Users/Batman/Desktop");
            expect(fs.rename).to.have.been.calledWith("/tmp/123", "/Users/Batman/Desktop/batman.png");
        });
    });
    
    it("stat should call stat and pass", function() {
        sandbox.stub(fs, "stat", FakeFS.stat).callsArgWith(1, null, {test: "statobject"});

        var promise = FileUtils.stat("/Users/Batman/Desktop/robin.png");
        return expect(promise).to.eventually.be.fulfilled.then(function(result) {
            expect(fs.stat).to.have.been.calledWith("/Users/Batman/Desktop/robin.png");
            expect(result).to.have.property("test", "statobject");
        });
    });
    
    it("stat should call stat and reject on error", function() {
        sandbox.stub(fs, "stat", FakeFS.stat).callsArgWith(1, "this is an error");

        var promise = FileUtils.stat("/Users/Batman/Desktop/robin.png");
        return expect(promise).to.eventually.be.rejectedWith("this is an error");
    });

    it("if moving a file fails, should fallback to a copy and delete", function() {
        sandbox.stub(fs, "mkdirs", FakeFS.mkdirs);
        sandbox.stub(fs, "rename", FakeFS.failingRename);
        sandbox.stub(fs, "copy", FakeFS.copy);
        sandbox.stub(fs, "remove", FakeFS.remove);

        var promise = FileUtils.moveTemporaryFile("/tmp/123", "/Users/Batman/Desktop/batman.png");
        return expect(promise).to.eventually.be.fulfilled.then(function() {
            expect(fs.copy).to.have.been.calledWith("/tmp/123", "/Users/Batman/Desktop/batman.png");
            expect(fs.remove).to.have.been.calledWith("/tmp/123");
        });
    });

    it("if moving a temporary file fails completely, should delete the temporary file and throw a write error", function() {
        sandbox.stub(fs, "mkdirs", FakeFS.mkdirs);
        sandbox.stub(fs, "rename", FakeFS.failingRename);
        // When renaming fails, make sure that copying won't work either!
        sandbox.stub(fs, "copy", FakeFS.failingCopy);
        sandbox.stub(fs, "remove", FakeFS.remove);

        var promise = FileUtils.moveTemporaryFile("/tmp/123", "/Users/Batman/Desktop/batman.png");
        return expect(promise).to.eventually.be.rejected.then(function() {
            var error = promise.inspect().reason;
            expect(error instanceof FileUtils.PermissionsWriteError);
            expect(error.message == "PermissionsWriteError");
            expect(fs.remove).to.have.been.calledWith("/tmp/123");
        });
    });

    it("should only show the file path once per generator session per path", function() {
        sandbox.stub(FileUtils, "openFolderInOS").returns(Q.resolve());
        var promise1 = FileUtils.openFolderOnceInOS("one/two");
        return expect(promise1).to.eventually.be.fulfilled.then(function() {
            expect(FileUtils.openFolderInOS).to.have.been.calledWith("one/two");
            var promise2 = FileUtils.openFolderOnceInOS("one/two");
            return expect(promise2).to.eventually.be.fulfilled.then(function() {
                expect(FileUtils.openFolderInOS).to.have.been.calledOnce;
                var promise3 = FileUtils.openFolderOnceInOS("three");
                return expect(promise3).to.eventually.be.fulfilled.then(function() {
                    expect(FileUtils.openFolderInOS).to.have.been.calledTwice;
                    expect(FileUtils.openFolderInOS).to.have.been.calledWith("three");
                });
            });
        });
    });
});
